const DEFAULT_API_URL = 'https://api.masszaps.com';

async function getStored() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['token', 'user', 'apiUrl'], (data) => resolve(data));
  });
}

async function getApiUrl() {
  const data = await getStored();
  return data.apiUrl || DEFAULT_API_URL;
}

async function getToken() {
  const data = await getStored();
  return data.token || null;
}

async function apiRequest(endpoint, options = {}) {
  const base = await getApiUrl();
  const token = await getToken();
  const url = base.replace(/\/$/, '') + endpoint;
  const headers = {
    'Content-Type': 'application/json',
    ...(token && { Authorization: `Bearer ${token}` }),
    ...(options.headers || {}),
  };
  const res = await fetch(url, { ...options, headers });
  const text = await res.text();
  const data = text ? JSON.parse(text) : {};
  if (!res.ok) {
    const msg = data.message || data.error || `Error ${res.status}`;
    throw new Error(msg);
  }
  return data;
}

export async function login(email, password) {
  const data = await apiRequest('/api/auth/login', {
    method: 'POST',
    body: JSON.stringify({ email, password }),
    headers: { 'Content-Type': 'application/json' },
  });
  if (data.status === 'success' && data.data?.token) {
    await new Promise((resolve) => {
      chrome.storage.local.set(
        { token: data.data.token, user: data.data.user },
        resolve
      );
    });
    return data.data;
  }
  throw new Error(data.message || 'Error al iniciar sesión');
}

export async function getInstances() {
  const data = await apiRequest('/api/instances', { method: 'GET' });
  return data.data ?? data ?? [];
}

export async function getQR(instanceName) {
  const data = await apiRequest(
    `/api/instances/${encodeURIComponent(instanceName)}/qr?force=1`,
    { method: 'GET' }
  );
  return data.data ?? data;
}

export async function logout() {
  return new Promise((resolve) => {
    chrome.storage.local.remove(['token', 'user'], resolve);
  });
}

export { getStored, getApiUrl, getToken };
